<?php
/**
 * 管理画面設定ページのビュー（移行機能付き）
 * 
 * @package Amazon_Affiliate_Block
 * @var array $settings 設定値の配列
 */

// 直接アクセスを防ぐ
if (!defined('ABSPATH')) {
    exit;
}

// 現在のタブを取得
$active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'general';

// 移行ツールのインスタンスを取得
$migration = null;
if (class_exists('AAB_Migration')) {
    $migration = new AAB_Migration();
}
?>

<div class="wrap">
    <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
    
    <!-- タブナビゲーション -->
    <nav class="nav-tab-wrapper">
        <a href="?page=aab-settings&tab=general" class="nav-tab <?php echo $active_tab == 'general' ? 'nav-tab-active' : ''; ?>">
            基本設定
        </a>
        <a href="?page=aab-settings&tab=api" class="nav-tab <?php echo $active_tab == 'api' ? 'nav-tab-active' : ''; ?>">
            API設定
        </a>
        <a href="?page=aab-settings&tab=migration" class="nav-tab <?php echo $active_tab == 'migration' ? 'nav-tab-active' : ''; ?>">
            AmazonJS移行
        </a>
        <a href="?page=aab-settings&tab=usage" class="nav-tab <?php echo $active_tab == 'usage' ? 'nav-tab-active' : ''; ?>">
            使い方
        </a>
    </nav>
    
    <div class="tab-content">
        <?php if ($active_tab == 'general'): ?>
            <!-- 基本設定タブ -->
            <?php settings_errors('aab_settings'); ?>
            
            <form method="post" action="">
                <?php wp_nonce_field('aab_settings_save', 'aab_nonce'); ?>
                
                <h2 class="title">基本設定</h2>
                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row">
                                <label for="affiliate_tag">Amazon.co.jp アフィリエイトタグ</label>
                            </th>
                            <td>
                                <input type="text" 
                                       id="affiliate_tag"
                                       name="affiliate_tag" 
                                       value="<?php echo esc_attr($settings['affiliate_tag']); ?>" 
                                       class="regular-text" 
                                       pattern="[a-z0-9\-]+" />
                                <p class="description">
                                    あなたのAmazonアソシエイトタグを入力してください（例：mysite-22）<br>
                                    ※英数字とハイフンのみ使用可能です
                                </p>
                            </td>
                        </tr>
                    </tbody>
                </table>
                
                <?php submit_button('設定を保存'); ?>
            </form>
            
        <?php elseif ($active_tab == 'api'): ?>
            <!-- API設定タブ -->
            <?php settings_errors('aab_settings'); ?>
            
            <form method="post" action="">
                <?php wp_nonce_field('aab_settings_save', 'aab_nonce'); ?>
                
                <h2 class="title">PA-API設定（オプション）</h2>
                
                <?php if ($settings['api_status'] === 'throttled'): ?>
                    <div class="notice notice-warning">
                        <p><strong>⚠️ API制限中</strong></p>
                        <p>Amazonアソシエイトで過去30日間の売上が3件未満のため、PA-APIが制限されています。</p>
                        <p>商品情報は手動で入力することで、プラグインは正常に動作します。</p>
                    </div>
                <?php endif; ?>
                
                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row">
                                <label for="api_key">PA-API アクセスキー</label>
                            </th>
                            <td>
                                <input type="text" 
                                       id="api_key"
                                       name="api_key" 
                                       value="<?php echo esc_attr($settings['api_key']); ?>" 
                                       class="regular-text" />
                                <p class="description">商品検索機能を使用する場合は入力してください</p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="secret_key">PA-API シークレットキー</label>
                            </th>
                            <td>
                                <input type="password" 
                                       id="secret_key"
                                       name="secret_key" 
                                       value="<?php echo esc_attr($settings['secret_key']); ?>" 
                                       class="regular-text" 
                                       autocomplete="new-password" />
                                <p class="description">PA-APIのシークレットキーを入力してください</p>
                            </td>
                        </tr>
                        
                        <?php if ($settings['api_configured']): ?>
                        <tr>
                            <th scope="row">API状態</th>
                            <td>
                                <?php if ($settings['rate_limit_active']): ?>
                                    <span style="color: orange;">⚠️ レート制限中（しばらくお待ちください）</span>
                                <?php elseif ($settings['api_status'] === 'throttled'): ?>
                                    <span style="color: red;">❌ 売上実績制限中</span>
                                <?php else: ?>
                                    <span style="color: green;">✅ 正常</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
                
                <?php submit_button('API設定を保存'); ?>
            </form>
            
        <?php elseif ($active_tab == 'migration'): ?>
            <!-- 移行タブ -->
            <?php
            // エラー表示
            settings_errors('aab_migration');
            
            // 移行処理
            if (isset($_POST['start_migration']) && check_admin_referer('aab_migration')) {
                if ($migration) {
                    $migration->execute_migration();
                    // 実行後のメッセージを表示
                    settings_errors('aab_migration');
                }
            }
            
            // スキャン処理
            $scan_results = null;
            if (isset($_POST['scan_content']) && check_admin_referer('aab_migration')) {
                if ($migration) {
                    try {
                        $scan_results = $migration->scan_for_amazonjs();
                    } catch (Exception $e) {
                        ?>
                        <div class="notice notice-error">
                            <p>スキャン中にエラーが発生しました: <?php echo esc_html($e->getMessage()); ?></p>
                        </div>
                        <?php
                    }
                }
            }
            ?>
            
            <h2 class="title">AmazonJS移行ツール</h2>
            
            <div class="notice notice-info">
                <p>このツールは、既存のAmazonJSショートコードをAmazon Affiliate Blockのショートコードに変換します。</p>
            </div>
            
            <!-- ステップ1: スキャン -->
            <div class="card">
                <h3>ステップ1: コンテンツのスキャン</h3>
                <p>まず、AmazonJSショートコードが使用されている投稿をスキャンします。</p>
                
                <form method="post" action="">
                    <?php wp_nonce_field('aab_migration'); ?>
                    <p>
                        <input type="submit" name="scan_content" class="button button-primary" value="コンテンツをスキャン">
                    </p>
                </form>
                
                <?php if ($scan_results !== null): ?>
                    <div class="scan-results" style="margin-top: 20px; padding: 15px; background: #f0f0f0; border-radius: 5px;">
                        <h4>スキャン結果</h4>
                        <p>検出されたAmazonJSショートコード: <strong><?php echo count($scan_results); ?>件</strong></p>
                        
                        <?php if (count($scan_results) > 0): ?>
                            <table class="wp-list-table widefat fixed striped" style="margin-top: 10px;">
                                <thead>
                                    <tr>
                                        <th style="width: 80px;">投稿ID</th>
                                        <th>タイトル</th>
                                        <th style="width: 100px;">検出数</th>
                                        <th>サンプル</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($scan_results as $result): ?>
                                        <tr>
                                            <td><?php echo $result['post_id']; ?></td>
                                            <td>
                                                <a href="<?php echo get_edit_post_link($result['post_id']); ?>" target="_blank">
                                                    <?php echo esc_html($result['title']); ?>
                                                </a>
                                            </td>
                                            <td><?php echo $result['count']; ?></td>
                                            <td><code style="font-size: 11px;"><?php echo esc_html(substr($result['sample'], 0, 50) . '...'); ?></code></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php else: ?>
                            <p style="color: green;">✅ AmazonJSショートコードは見つかりませんでした。移行の必要はありません。</p>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- ステップ2: 移行実行 -->
            <div class="card" style="margin-top: 20px;">
                <h3>ステップ2: 移行実行</h3>
                
                <p>以下のショートコードが変換されます：</p>
                <ul style="list-style: disc; margin-left: 20px;">
                    <li><code>[amazonjs]</code> → <code>[amazon_link]</code></li>
                    <li><code>[amazon]</code> → <code>[amazon_link]</code></li>
                    <li><code>[amazon_link]</code> (AmazonJS版) → <code>[amazon_link]</code> (AAB版)</li>
                </ul>
                
                <div class="notice notice-warning">
                    <p><strong>重要:</strong> 実行前に必ずデータベースのバックアップを取ってください。</p>
                </div>
                
                <form method="post" action="">
                    <?php wp_nonce_field('aab_migration'); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th>移行オプション</th>
                            <td>
                                <label>
                                    <input type="checkbox" name="create_backup" value="1" checked>
                                    変換前のコンテンツをバックアップ（リビジョンとして保存）
                                </label><br>
                                
                                <label>
                                    <input type="checkbox" name="dry_run" value="1">
                                    ドライラン（実際には変換しない）
                                </label>
                            </td>
                        </tr>
                    </table>
                    
                    <p>
                        <input type="submit" name="start_migration" class="button button-primary" value="移行を開始" 
                               onclick="return confirm('本当に移行を実行しますか？');">
                    </p>
                </form>
            </div>
            
            <!-- 互換モード -->
            <div class="card" style="margin-top: 20px;">
                <h3>互換モード</h3>
                <p>AmazonJSのショートコードをそのまま使えるようにする互換モードを有効にできます。</p>
                
                <form method="post" action="">
                    <?php wp_nonce_field('aab_settings_save', 'aab_nonce'); ?>
                    <table class="form-table">
                        <tr>
                            <th>互換モード</th>
                            <td>
                                <label>
                                    <input type="checkbox" name="auto_convert_amazonjs" value="1" 
                                           <?php checked(get_option('aab_auto_convert_amazonjs'), 1); ?>>
                                    AmazonJSショートコードを自動的に処理する
                                </label>
                                <p class="description">
                                    有効にすると、既存のAmazonJSショートコードがそのまま動作します。<br>
                                    ※移行が完了したら無効にすることをお勧めします。
                                </p>
                            </td>
                        </tr>
                    </table>
                    <?php submit_button('互換モード設定を保存'); ?>
                </form>
            </div>
            
        <?php elseif ($active_tab == 'usage'): ?>
            <!-- 使い方タブ -->
            <div class="aab-usage-guide">
                <h2>使い方</h2>
                
                <h3>基本的な使い方</h3>
                <ol>
                    <li>「基本設定」タブでアフィリエイトタグを設定</li>
                    <li>投稿編集画面で「Amazon商品」ブロックを追加</li>
                    <li>ASINコードを入力して商品リンクを作成</li>
                </ol>
                
                <h3>ショートコード</h3>
                <p>クラシックエディタでは以下のショートコードが使用できます：</p>
                <div class="code-example">
                    <code>[amazon_link asin="B08XXXXX" title="商品名"]</code>
                </div>
                
                <h4>利用可能なパラメータ：</h4>
                <ul>
                    <li><strong>asin</strong> - Amazon商品のASINコード（必須）</li>
                    <li><strong>title</strong> - リンクテキスト（デフォルト: "商品リンク"）</li>
                    <li><strong>type</strong> - 表示タイプ（text/card/image）</li>
                    <li><strong>marketplace</strong> - マーケットプレイス（co.jp/com/co.uk/de/fr）</li>
                    <li><strong>image</strong> - 画像URL（カード/画像表示時）</li>
                    <li><strong>price</strong> - 価格表示（カード表示時）</li>
                </ul>
                
                <h3>PA-API利用時の注意事項</h3>
                <ul>
                    <li>API制限を避けるため、商品情報は24時間キャッシュされます</li>
                    <li>検索結果は1時間キャッシュされます</li>
                    <li>エラーが発生した場合は、手動で商品情報を入力してください</li>
                    <li>制限に達した場合は、5分程度待ってから再試行してください</li>
                </ul>
                
                <h3>AmazonJSからの移行</h3>
                <p>「AmazonJS移行」タブから、既存のAmazonJSショートコードを自動変換できます。</p>
                <ol>
                    <li>コンテンツをスキャンして影響範囲を確認</li>
                    <li>バックアップを有効にして移行実行</li>
                    <li>必要に応じて互換モードを有効化</li>
                </ol>
            </div>
        <?php endif; ?>
    </div>
    
    <?php if ($active_tab == 'general'): ?>
    <hr style="margin-top: 40px;">
    
    <div style="color: #666; font-size: 12px; line-height: 1.6;">
        <p>※ このプラグインは無料でご利用いただけます。開発継続のため、生成されるリンクの一部に開発者のアフィリエイトタグが含まれる場合があります。</p>
    </div>
    <?php endif; ?>
</div>

<style>
.nav-tab-wrapper {
    margin-bottom: 20px;
}

.tab-content {
    background: #fff;
    padding: 20px;
    border: 1px solid #ccd0d4;
    border-top: none;
}

.card {
    background: #fff;
    border: 1px solid #ccd0d4;
    padding: 20px;
    margin-top: 20px;
}

.card h3 {
    margin-top: 0;
    border-bottom: 1px solid #eee;
    padding-bottom: 10px;
}

.aab-usage-guide {
    max-width: 800px;
}

.aab-usage-guide h3 {
    margin-top: 30px;
    color: #23282d;
}

.code-example {
    background: #f1f1f1;
    padding: 10px;
    border-radius: 3px;
    margin: 10px 0;
}

.code-example code {
    background: none;
    font-size: 14px;
}

.scan-results table {
    margin-top: 15px;
}

.scan-results table th {
    font-weight: 600;
}
</style>