<?php
/**
 * ショートコード管理クラス
 */
class AAB_Shortcode {
    
    private $affiliate;
    
    /**
     * コンストラクタ
     */
    public function __construct($affiliate) {
        $this->affiliate = $affiliate;
        add_shortcode('amazon_link', array($this, 'render'));
    }
    
    /**
     * ショートコードのレンダリング
     */
    public function render($atts) {
        try {
            $atts = shortcode_atts(array(
                'asin' => '',
                'title' => '商品リンク',
                'type' => 'text',
                'marketplace' => 'co.jp',
                'image' => '',
                'price' => ''
            ), $atts, 'amazon_link');
            
            // ASIN検証
            $asin = AAB_Helper::validate_asin($atts['asin']);
            if (!$asin) {
                AAB_Helper::log_error('ショートコード: 無効なASIN', array('asin' => $atts['asin']));
                return '';
            }
            
            // アフィリエイトタグ取得
            $affiliate_tag = $this->affiliate->get_affiliate_tag();
            if (empty($affiliate_tag)) {
                AAB_Helper::log_error('ショートコード: アフィリエイトタグ未設定');
                return '';
            }
            
            // Amazon URL生成
            $amazon_url = AAB_Helper::generate_amazon_url($asin, $affiliate_tag, $atts['marketplace']);
            
            // スタイルを読み込む
            wp_enqueue_style('aab-frontend-style');
            
            // タイプに応じてレンダリング
            switch ($atts['type']) {
                case 'card':
                    return $this->render_card($amazon_url, $atts);
                case 'image':
                    return $this->render_image($amazon_url, $atts);
                case 'text':
                default:
                    return $this->render_text($amazon_url, $atts['title']);
            }
            
        } catch (Exception $e) {
            AAB_Helper::log_error('ショートコード処理エラー', array('error' => $e->getMessage()));
            return '';
        }
    }
    
    /**
     * カード形式のレンダリング
     */
    private function render_card($url, $atts) {
        $renderer = new AAB_Block_Renderer();
        $attributes = array(
            'title' => $atts['title'],
            'imageUrl' => $atts['image'],
            'price' => $atts['price'],
            'linkType' => 'card'
        );
        return $renderer->render($url, $attributes);
    }
    
    /**
     * 画像形式のレンダリング
     */
    private function render_image($url, $atts) {
        if (empty($atts['image'])) {
            return $this->render_text($url, $atts['title']);
        }
        
        return sprintf(
            '<div class="aab-product"><a href="%s" target="_blank" rel="noopener noreferrer"><img src="%s" alt="%s" class="aab-product-image-only" loading="lazy"></a></div>',
            esc_url($url),
            esc_url($atts['image']),
            esc_attr($atts['title'])
        );
    }
    
    /**
     * テキスト形式のレンダリング
     */
    private function render_text($url, $title) {
        return sprintf(
            '<a href="%s" target="_blank" rel="noopener noreferrer" class="aab-product-text-link">%s</a>',
            esc_url($url),
            esc_html($title)
        );
    }
}