<?php
/**
 * AmazonJS移行ツールクラス（修正版）
 * 
 * 既存のAmazonJSショートコードを検出して変換
 */
class AAB_Migration {
    
    private $converted_count = 0;
    private $migration_log = array();
    
    /**
     * コンストラクタ
     */
    public function __construct() {
        // 互換モードの自動有効化
        if (get_option('aab_auto_convert_amazonjs', false)) {
            $this->register_compatibility_shortcodes();
        }
    }
    
    /**
     * AmazonJSショートコードをスキャン（公開メソッド）
     */
    public function scan_for_amazonjs() {
        global $wpdb;
        
        $results = array();
        
        try {
            // シンプルなLIKEクエリを使用
            $query = "SELECT ID, post_title, post_content 
                      FROM {$wpdb->posts} 
                      WHERE (post_content LIKE '%[amazonjs%' 
                         OR post_content LIKE '%[amazon %' 
                         OR post_content LIKE '%[amazon_link%')
                      AND post_status IN ('publish', 'draft', 'private')
                      AND post_type IN ('post', 'page')";
            
            $posts = $wpdb->get_results($query);
            
            if (!$posts) {
                return array();
            }
            
            foreach ($posts as $post) {
                $shortcode_count = 0;
                $samples = array();
                
                // [amazonjs] パターン
                if (preg_match_all('/\[amazonjs[^\]]*\]/i', $post->post_content, $matches)) {
                    $shortcode_count += count($matches[0]);
                    if (!empty($matches[0][0])) {
                        $samples[] = $matches[0][0];
                    }
                }
                
                // [amazon ] パターン
                if (preg_match_all('/\[amazon\s+[^\]]*\]/i', $post->post_content, $matches)) {
                    $shortcode_count += count($matches[0]);
                    if (!empty($matches[0][0]) && count($samples) == 0) {
                        $samples[] = $matches[0][0];
                    }
                }
                
                // [amazon_link] パターン（AmazonJS版のみ）
                if (preg_match_all('/\[amazon_link[^\]]*\]/i', $post->post_content, $matches)) {
                    foreach ($matches[0] as $match) {
                        if (strpos($match, 'asins=') !== false || strpos($match, 'template=') !== false) {
                            $shortcode_count++;
                            if (count($samples) == 0) {
                                $samples[] = $match;
                            }
                        }
                    }
                }
                
                if ($shortcode_count > 0) {
                    $results[] = array(
                        'post_id' => $post->ID,
                        'title' => $post->post_title,
                        'count' => $shortcode_count,
                        'sample' => !empty($samples) ? $samples[0] : '',
                        'type' => 'mixed'
                    );
                }
            }
            
        } catch (Exception $e) {
            AAB_Helper::log_error('スキャンエラー', array('error' => $e->getMessage()));
        }
        
        return $results;
    }
    
    /**
     * 移行を実行（公開メソッド）
     */
    public function execute_migration() {
        try {
            $dry_run = isset($_POST['dry_run']) && $_POST['dry_run'];
            $create_backup = isset($_POST['create_backup']) && $_POST['create_backup'];
            
            $scan_results = $this->scan_for_amazonjs();
            
            if (empty($scan_results)) {
                add_settings_error(
                    'aab_migration',
                    'no_content',
                    '変換対象のショートコードが見つかりませんでした。',
                    'info'
                );
                return;
            }
            
            $this->converted_count = 0;
            $this->migration_log = array();
            
            foreach ($scan_results as $result) {
                $post = get_post($result['post_id']);
                
                if (!$post) {
                    continue;
                }
                
                // バックアップ作成
                if ($create_backup && !$dry_run) {
                    wp_save_post_revision($result['post_id']);
                }
                
                // コンテンツを変換
                $new_content = $this->convert_content($post->post_content);
                
                if ($new_content !== $post->post_content) {
                    if (!$dry_run) {
                        wp_update_post(array(
                            'ID' => $result['post_id'],
                            'post_content' => $new_content
                        ));
                    }
                    
                    $this->converted_count++;
                    $this->migration_log[] = sprintf(
                        '投稿ID %d: %s を変換%s',
                        $result['post_id'],
                        $result['title'],
                        $dry_run ? '（ドライラン）' : '完了'
                    );
                }
            }
            
            if ($this->converted_count > 0) {
                add_settings_error(
                    'aab_migration',
                    'migration_complete',
                    sprintf('%d件の投稿を変換しました%s', $this->converted_count, $dry_run ? '（ドライラン）' : ''),
                    'success'
                );
            } else {
                add_settings_error(
                    'aab_migration',
                    'no_changes',
                    '変換が必要なショートコードはありませんでした。',
                    'info'
                );
            }
            
        } catch (Exception $e) {
            add_settings_error(
                'aab_migration',
                'migration_error',
                'エラーが発生しました: ' . $e->getMessage(),
                'error'
            );
        }
    }
    
    /**
     * コンテンツ内のショートコードを変換
     */
    private function convert_content($content) {
        // [amazonjs] の変換
        $content = preg_replace_callback(
            '/\[amazonjs([^\]]*)\]/i',
            array($this, 'convert_amazonjs_shortcode'),
            $content
        );
        
        // [amazon] の変換
        $content = preg_replace_callback(
            '/\[amazon\s+([^\]]*)\]/i',
            array($this, 'convert_amazon_shortcode'),
            $content
        );
        
        // [amazon_link] (旧形式) の変換
        $content = preg_replace_callback(
            '/\[amazon_link([^\]]*)\]/i',
            array($this, 'convert_amazon_link_shortcode'),
            $content
        );
        
        return $content;
    }
    
    /**
     * [amazonjs] ショートコードを変換
     */
    private function convert_amazonjs_shortcode($matches) {
        $attrs_string = isset($matches[1]) ? $matches[1] : '';
        $attrs = shortcode_parse_atts($attrs_string);
        
        if (!is_array($attrs)) {
            return $matches[0];
        }
        
        $new_attrs = array();
        
        // ASIN
        if (isset($attrs['asin'])) {
            $new_attrs['asin'] = $attrs['asin'];
        }
        
        // タイトル
        if (isset($attrs['title'])) {
            $new_attrs['title'] = $attrs['title'];
        }
        
        // ロケール → マーケットプレイス
        if (isset($attrs['locale'])) {
            $marketplace_map = array(
                'JP' => 'co.jp',
                'US' => 'com',
                'UK' => 'co.uk',
                'DE' => 'de',
                'FR' => 'fr'
            );
            
            $locale = strtoupper($attrs['locale']);
            if (isset($marketplace_map[$locale])) {
                $new_attrs['marketplace'] = $marketplace_map[$locale];
            }
        }
        
        // 表示タイプ
        if (isset($attrs['tmpl'])) {
            $type_map = array(
                'Small' => 'text',
                'Medium' => 'card',
                'Large' => 'card'
            );
            
            if (isset($type_map[$attrs['tmpl']])) {
                $new_attrs['type'] = $type_map[$attrs['tmpl']];
            }
        }
        
        // 新しいショートコードを生成
        if (empty($new_attrs['asin'])) {
            return $matches[0]; // ASINがない場合は変換しない
        }
        
        $new_shortcode = '[amazon_link';
        foreach ($new_attrs as $key => $value) {
            $new_shortcode .= sprintf(' %s="%s"', $key, esc_attr($value));
        }
        $new_shortcode .= ']';
        
        return $new_shortcode;
    }
    
    /**
     * [amazon] ショートコードを変換
     */
    private function convert_amazon_shortcode($matches) {
        return $this->convert_amazonjs_shortcode($matches);
    }
    
    /**
     * [amazon_link] (旧形式) を変換
     */
    private function convert_amazon_link_shortcode($matches) {
        $attrs_string = isset($matches[1]) ? $matches[1] : '';
        $attrs = shortcode_parse_atts($attrs_string);
        
        if (!is_array($attrs)) {
            return $matches[0];
        }
        
        // AmazonJS形式でない場合はそのまま
        if (!isset($attrs['asins']) && !isset($attrs['template'])) {
            return $matches[0];
        }
        
        $new_attrs = array();
        
        // asins → asin
        if (isset($attrs['asins'])) {
            $asins = explode(',', $attrs['asins']);
            $new_attrs['asin'] = trim($asins[0]);
        } elseif (isset($attrs['asin'])) {
            $new_attrs['asin'] = $attrs['asin'];
        }
        
        // template → type
        if (isset($attrs['template'])) {
            $template_map = array(
                'ProductCarousel' => 'card',
                'ProductGrid' => 'card',
                'ProductLink' => 'text'
            );
            
            if (isset($template_map[$attrs['template']])) {
                $new_attrs['type'] = $template_map[$attrs['template']];
            }
        }
        
        // タイトル
        if (isset($attrs['title'])) {
            $new_attrs['title'] = $attrs['title'];
        }
        
        // ASINがない場合は変換しない
        if (empty($new_attrs['asin'])) {
            return $matches[0];
        }
        
        // 新しいショートコードを生成
        $new_shortcode = '[amazon_link';
        foreach ($new_attrs as $key => $value) {
            $new_shortcode .= sprintf(' %s="%s"', $key, esc_attr($value));
        }
        $new_shortcode .= ']';
        
        return $new_shortcode;
    }
    
    /**
     * 互換性ショートコードを登録
     */
    public function register_compatibility_shortcodes() {
        // AmazonJSのショートコードをフック
        add_shortcode('amazonjs', array($this, 'handle_amazonjs_shortcode'));
        add_shortcode('amazon', array($this, 'handle_amazonjs_shortcode'));
    }
    
    /**
     * AmazonJSショートコードを処理
     */
    public function handle_amazonjs_shortcode($atts) {
        // パラメータを変換
        $converted = array();
        
        // ASIN
        if (isset($atts['asin'])) {
            $converted['asin'] = $atts['asin'];
        }
        
        // タイトル
        if (isset($atts['title'])) {
            $converted['title'] = $atts['title'];
        }
        
        // ロケール
        if (isset($atts['locale'])) {
            $locale_map = array(
                'JP' => 'co.jp',
                'US' => 'com',
                'UK' => 'co.uk'
            );
            $locale = strtoupper($atts['locale']);
            $converted['marketplace'] = isset($locale_map[$locale]) ? $locale_map[$locale] : 'co.jp';
        }
        
        // テンプレート
        if (isset($atts['tmpl'])) {
            $tmpl_map = array(
                'Small' => 'text',
                'Medium' => 'card',
                'Large' => 'card'
            );
            $converted['type'] = isset($tmpl_map[$atts['tmpl']]) ? $tmpl_map[$atts['tmpl']] : 'text';
        }
        
        // AABのショートコードとして処理
        if (class_exists('AAB_Shortcode') && class_exists('AAB_Affiliate')) {
            $shortcode = new AAB_Shortcode(new AAB_Affiliate());
            return $shortcode->render($converted);
        }
        
        return '';
    }
}