<?php
/**
 * コアクラス - プラグインの初期化と依存関係管理
 */
class AAB_Core {
    
    private static $instance = null;
    private $components = array();
    
    /**
     * シングルトンインスタンス取得
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * コンストラクタ
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }
    
    /**
     * 依存関係の読み込み
     */
    private function load_dependencies() {
        // ヘルパークラス（最初に読み込む）
        require_once AAB_PLUGIN_DIR . 'includes/class-aab-helper.php';
        
        // API関連クラス（apiサブディレクトリ）
        $this->load_api_classes();
        
        // メインコンポーネント
        $this->load_main_components();
        
        // インスタンス化
        $this->initialize_components();
    }
    
    /**
     * API関連クラスの読み込み
     */
    private function load_api_classes() {
        $api_dir = AAB_PLUGIN_DIR . 'includes/api/';
        
        // APIディレクトリが存在するか確認
        if (!is_dir($api_dir)) {
            // ディレクトリがない場合は作成を試みる
            wp_mkdir_p($api_dir);
            AAB_Helper::log_error('APIディレクトリが存在しません', array('path' => $api_dir));
        }
        
        // API関連クラスを順番に読み込み
        $api_classes = array(
            'class-aab-marketplace-manager.php',
            'class-aab-error-handler.php',
            'class-aab-product-parser.php',
            'class-aab-api-signature.php',
            'class-aab-api-client.php'
        );
        
        foreach ($api_classes as $class_file) {
            $file_path = $api_dir . $class_file;
            if (file_exists($file_path)) {
                require_once $file_path;
            } else {
                // ファイルが存在しない場合の警告
                AAB_Helper::log_error('APIクラスファイルが見つかりません', array(
                    'file' => $class_file,
                    'path' => $file_path
                ));
            }
        }
    }
    
    /**
     * メインコンポーネントの読み込み
     */
    private function load_main_components() {
        $includes_dir = AAB_PLUGIN_DIR . 'includes/';
        
        // 必須コンポーネント
        $required_components = array(
            'class-aab-affiliate.php',
            'class-aab-api.php',
            'class-aab-block.php',
            'class-aab-block-renderer.php',
            'class-aab-shortcode.php',
            'class-aab-admin.php'
        );
        
        foreach ($required_components as $component) {
            $file_path = $includes_dir . $component;
            if (file_exists($file_path)) {
                require_once $file_path;
            } else {
                AAB_Helper::log_error('コンポーネントファイルが見つかりません', array(
                    'file' => $component,
                    'path' => $file_path
                ));
            }
        }
    }
    
    /**
     * コンポーネントの初期化
     */
    private function initialize_components() {
        try {
            // ヘルパーは既にstaticクラスなのでインスタンス化不要
            $this->components['helper'] = 'AAB_Helper';
            
            // アフィリエイト管理
            if (class_exists('AAB_Affiliate')) {
                $this->components['affiliate'] = new AAB_Affiliate();
            }
            
            // API管理（新しい構造に対応）
            if (class_exists('AAB_Api')) {
                $this->components['api'] = new AAB_Api();
            }
            
            // ブロック管理
            if (class_exists('AAB_Block') && isset($this->components['affiliate'])) {
                $this->components['block'] = new AAB_Block($this->components['affiliate']);
            }
            
            // ショートコード管理
            if (class_exists('AAB_Shortcode') && isset($this->components['affiliate'])) {
                $this->components['shortcode'] = new AAB_Shortcode($this->components['affiliate']);
            }
            
            // 管理画面（管理者のみ）
            if (is_admin() && class_exists('AAB_Admin')) {
                $this->components['admin'] = new AAB_Admin();
            }
            
            // 移行ツール管理（管理者のみ）
            if (is_admin() && class_exists('AAB_Migration')) {
                $this->components['migration'] = new AAB_Migration();
            }
            
        } catch (Exception $e) {
            AAB_Helper::log_error('コンポーネント初期化エラー', array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * フックの初期化
     */
    private function init_hooks() {
        // 多言語対応
        add_action('init', array($this, 'load_textdomain'));
        
        // 管理画面通知
        add_action('admin_notices', array($this, 'display_admin_notices'));
        
        // プラグインアップデート時の処理
        add_action('upgrader_process_complete', array($this, 'upgrade_process'), 10, 2);
        
        // スタイルとスクリプトの登録
        add_action('init', array($this, 'register_assets'));
    }
    
    /**
     * 翻訳ファイルの読み込み
     */
    public function load_textdomain() {
        load_plugin_textdomain(
            'amazon-affiliate-block',
            false,
            dirname(plugin_basename(AAB_PLUGIN_FILE)) . '/languages'
        );
    }
    
    /**
     * アセットの登録
     */
    public function register_assets() {
        // フロントエンド用スタイル
        if (file_exists(AAB_PLUGIN_DIR . 'block/style.css')) {
            wp_register_style(
                'aab-frontend-style',
                AAB_PLUGIN_URL . 'block/style.css',
                array(),
                filemtime(AAB_PLUGIN_DIR . 'block/style.css')
            );
        }
    }
    
    /**
     * 管理画面通知
     */
    public function display_admin_notices() {
        // エラーメッセージの表示
        $errors = AAB_Helper::get_errors();
        if (!empty($errors)) {
            foreach ($errors as $error) {
                echo '<div class="notice notice-error is-dismissible">';
                echo '<p><strong>Amazon Affiliate Block:</strong> ' . esc_html($error) . '</p>';
                echo '</div>';
            }
            AAB_Helper::clear_errors();
        }
        
        // アップデート通知
        $this->display_update_notice();
    }
    
    /**
     * アップデート通知の表示
     */
    private function display_update_notice() {
        $current_version = AAB_PLUGIN_VERSION;
        $saved_version = get_option('aab_version', '0.0.0');
        
        if (version_compare($current_version, $saved_version, '>')) {
            // バージョンアップ時の処理
            $this->run_upgrade_routines($saved_version, $current_version);
            update_option('aab_version', $current_version);
            
            // アップデート完了通知
            if (current_user_can('manage_options')) {
                echo '<div class="notice notice-success is-dismissible">';
                echo '<p><strong>Amazon Affiliate Block:</strong> ';
                echo sprintf('バージョン %s にアップデートされました。', esc_html($current_version));
                echo '</p></div>';
            }
        }
    }
    
    /**
     * アップグレード処理
     */
    public function upgrade_process($upgrader_object, $options) {
        if ($options['action'] == 'update' && $options['type'] == 'plugin') {
            if (isset($options['plugins'])) {
                foreach ($options['plugins'] as $plugin) {
                    if ($plugin == AAB_PLUGIN_BASENAME) {
                        $this->clear_all_caches();
                    }
                }
            }
        }
    }
    
    /**
     * バージョンアップ時のルーチン
     */
    private function run_upgrade_routines($old_version, $new_version) {
        // 1.0.0 から 2.0.0 へのアップグレード
        if (version_compare($old_version, '2.0.0', '<')) {
            $this->upgrade_to_2_0_0();
        }
        
        // キャッシュクリア
        $this->clear_all_caches();
    }
    
    /**
     * バージョン2.0.0へのアップグレード処理
     */
    private function upgrade_to_2_0_0() {
        // 既存の設定をマイグレーション
        $existing_tag = get_option('aab_affiliate_tag');
        $existing_api_key = get_option('aab_api_key');
        $existing_secret_key = get_option('aab_secret_key');
        
        // 日本市場の設定として保存
        if ($existing_tag || $existing_api_key || $existing_secret_key) {
            update_option('aab_affiliate_tag_jp', $existing_tag);
            update_option('aab_api_key_jp', $existing_api_key);
            update_option('aab_secret_key_jp', $existing_secret_key);
            
            // デフォルトマーケットプレイスを日本に設定
            update_option('aab_default_marketplace', 'jp');
        }
    }
    
    /**
     * すべてのキャッシュをクリア
     */
    private function clear_all_caches() {
        global $wpdb;
        
        // AAB関連のトランジェントを削除
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_aab_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_aab_%'");
        
        AAB_Helper::log_error('キャッシュクリア完了', array(
            'timestamp' => current_time('mysql')
        ));
    }
    
    /**
     * コンポーネントを取得
     */
    public function get_component($name) {
        return isset($this->components[$name]) ? $this->components[$name] : null;
    }
    
    /**
     * プラグイン有効化
     */
    public static function activate() {
        // バージョン情報を保存
        add_option('aab_version', AAB_PLUGIN_VERSION);
        
        // デフォルト設定の追加
        add_option('aab_affiliate_tag', '');
        add_option('aab_default_marketplace', 'jp');
        add_option('aab_developer_support', true);
        
        // データベーステーブルの作成（必要に応じて）
        self::create_database_tables();
        
        // リライトルールをフラッシュ
        flush_rewrite_rules();
        
        // 有効化ログ
        if (class_exists('AAB_Helper')) {
            AAB_Helper::log_error('プラグイン有効化', array(
                'version' => AAB_PLUGIN_VERSION,
                'timestamp' => current_time('mysql')
            ));
        }
    }
    
    /**
     * プラグイン無効化
     */
    public static function deactivate() {
        // スケジュールされたイベントをクリア
        wp_clear_scheduled_hook('aab_daily_cleanup');
        
        // 一時的なオプションをクリア
        delete_transient('aab_api_rate_limit');
        
        // リライトルールをフラッシュ
        flush_rewrite_rules();
        
        // 無効化ログ
        if (class_exists('AAB_Helper')) {
            AAB_Helper::log_error('プラグイン無効化', array(
                'timestamp' => current_time('mysql')
            ));
        }
    }
    
    /**
     * プラグインアンインストール
     */
    public static function uninstall() {
        // オプションの削除（ユーザーの選択に応じて）
        if (get_option('aab_delete_data_on_uninstall', false)) {
            // すべてのオプションを削除
            $option_names = array(
                'aab_version',
                'aab_affiliate_tag',
                'aab_api_key',
                'aab_secret_key',
                'aab_default_marketplace',
                'aab_developer_support'
            );
            
            // マーケットプレイスごとの設定も削除
            $marketplaces = array('jp', 'us', 'uk', 'de', 'fr', 'it', 'es', 'ca', 'au', 'in');
            foreach ($marketplaces as $mp) {
                $option_names[] = 'aab_affiliate_tag_' . $mp;
                $option_names[] = 'aab_api_key_' . $mp;
                $option_names[] = 'aab_secret_key_' . $mp;
                $option_names[] = 'aab_api_status_' . $mp;
            }
            
            foreach ($option_names as $option) {
                delete_option($option);
            }
            
            // トランジェントも削除
            global $wpdb;
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_aab_%'");
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_aab_%'");
        }
    }
    
    /**
     * データベーステーブルの作成（将来の拡張用）
     */
    private static function create_database_tables() {
        // 将来的に統計情報などを保存する場合に使用
        // global $wpdb;
        // $charset_collate = $wpdb->get_charset_collate();
        // $table_name = $wpdb->prefix . 'aab_statistics';
        // ...
    }
}