<?php
/**
 * ブロック管理クラス
 */
class AAB_Block {
    
    private $affiliate;
    
    /**
     * コンストラクタ
     */
    public function __construct($affiliate) {
        $this->affiliate = $affiliate;
        add_action('init', array($this, 'register_block'));
    }
    
    /**
     * ブロックの登録
     */
    public function register_block() {
        try {
            $editor_script_path = AAB_PLUGIN_DIR . 'block/editor.js';
            $style_path = AAB_PLUGIN_DIR . 'block/style.css';
            
            // スクリプトの登録
            if (file_exists($editor_script_path)) {
                wp_register_script(
                    'aab-editor',
                    AAB_PLUGIN_URL . 'block/editor.js',
                    array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-data'),
                    filemtime($editor_script_path)
                );
            } else {
                AAB_Helper::log_error('editor.jsが見つかりません', array('path' => $editor_script_path));
            }
            
            // スタイルの登録
            if (file_exists($style_path)) {
                wp_register_style(
                    'aab-style',
                    AAB_PLUGIN_URL . 'block/style.css',
                    array(),
                    filemtime($style_path)
                );
                
                wp_register_style(
                    'aab-frontend-style',
                    AAB_PLUGIN_URL . 'block/style.css',
                    array(),
                    filemtime($style_path)
                );
            } else {
                AAB_Helper::log_error('style.cssが見つかりません', array('path' => $style_path));
            }
            
            // ブロックタイプの登録
            register_block_type('aab/amazon-product', array(
                'editor_script' => 'aab-editor',
                'editor_style' => 'aab-style',
                'style' => 'aab-frontend-style',
                'render_callback' => array($this, 'render'),
                'attributes' => $this->get_block_attributes()
            ));
            
        } catch (Exception $e) {
            AAB_Helper::log_error('ブロック登録エラー', array('error' => $e->getMessage()));
        }
    }
    
    /**
     * ブロック属性の定義
     */
    private function get_block_attributes() {
        return array(
            'asin' => array('type' => 'string', 'default' => ''),
            'title' => array('type' => 'string', 'default' => ''),
            'imageUrl' => array('type' => 'string', 'default' => ''),
            'price' => array('type' => 'string', 'default' => ''),
            'linkType' => array('type' => 'string', 'default' => 'text'),
            'marketplace' => array('type' => 'string', 'default' => 'co.jp')
        );
    }
    
    /**
     * ブロックのレンダリング
     */
    public function render($attributes) {
        try {
            // 属性の取得
            $asin = isset($attributes['asin']) ? AAB_Helper::validate_asin($attributes['asin']) : false;
            if (!$asin) {
                AAB_Helper::log_error('無効なASIN', array('asin' => $attributes['asin'] ?? ''));
                return '<!-- AAB: 無効なASIN -->';
            }
            
            $affiliate_tag = $this->affiliate->get_affiliate_tag();
            if (empty($affiliate_tag)) {
                AAB_Helper::log_error('アフィリエイトタグ未設定');
                return '<!-- AAB: アフィリエイトタグ未設定 -->';
            }
            
            $marketplace = isset($attributes['marketplace']) ? $attributes['marketplace'] : 'co.jp';
            $amazon_url = AAB_Helper::generate_amazon_url($asin, $affiliate_tag, $marketplace);
            
            // 画像URLの検証
            if (!empty($attributes['imageUrl']) && !AAB_Helper::validate_url($attributes['imageUrl'])) {
                AAB_Helper::log_error('無効な画像URL', array('url' => $attributes['imageUrl']));
                $attributes['imageUrl'] = '';
            }
            
            // HTMLレンダラーのインスタンス化と実行
            $renderer = new AAB_Block_Renderer();
            return $renderer->render($amazon_url, $attributes);
            
        } catch (Exception $e) {
            AAB_Helper::log_error('ブロックレンダリングエラー', array('error' => $e->getMessage()));
            return '<!-- AAB: レンダリングエラー -->';
        }
    }
}