<?php
/**
 * Amazon PA-API v5 メインクラス
 * 
 * APIの初期化とAJAXハンドラーの管理を担当
 */
class AAB_Api {
    
    private $client;
    private $marketplace_manager;
    private $error_handler;
    private $api_status = null;
    
    /**
     * コンストラクタ
     */
    public function __construct() {
        // 依存クラスの読み込み
        $this->load_dependencies();
        
        // マネージャーの初期化
        $this->marketplace_manager = new AAB_Marketplace_Manager();
        $this->error_handler = new AAB_Error_Handler();
        $this->client = new AAB_Api_Client($this->marketplace_manager, $this->error_handler);
        
        // API状態を取得
        $this->api_status = get_option('aab_api_status', null);
        
        // フックの登録
        $this->init_hooks();
    }
    
    /**
     * 依存クラスの読み込み
     */
    private function load_dependencies() {
        require_once AAB_PLUGIN_DIR . 'includes/api/class-aab-marketplace-manager.php';
        require_once AAB_PLUGIN_DIR . 'includes/api/class-aab-api-client.php';
        require_once AAB_PLUGIN_DIR . 'includes/api/class-aab-api-signature.php';
        require_once AAB_PLUGIN_DIR . 'includes/api/class-aab-error-handler.php';
        require_once AAB_PLUGIN_DIR . 'includes/api/class-aab-product-parser.php';
    }
    
    /**
     * フックの初期化
     */
    private function init_hooks() {
        // AJAXハンドラー
        add_action('wp_ajax_aab_get_product_info', array($this, 'ajax_get_product_info'));
        add_action('wp_ajax_aab_search_products', array($this, 'ajax_search_products'));
        add_action('wp_ajax_aab_test_api', array($this, 'ajax_test_api'));
        add_action('wp_ajax_aab_get_marketplace_status', array($this, 'ajax_get_marketplace_status'));
        
        // 管理画面
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('admin_notices', array($this, 'display_api_status_notice'));
    }
    
    /**
     * APIが設定されているかチェック
     */
    public function is_configured($marketplace_code = null) {
        return $this->marketplace_manager->is_configured($marketplace_code);
    }
    
    /**
     * 商品情報を取得
     */
    public function get_product_info($asin, $marketplace_code = null) {
        // キャッシュチェック
        $cache_key = $this->get_cache_key('product', $asin, $marketplace_code);
        $cached_data = get_transient($cache_key);
        if ($cached_data !== false) {
            return $cached_data;
        }
        
        // API呼び出し
        $result = $this->client->get_product($asin, $marketplace_code);
        
        if (!is_wp_error($result)) {
            // キャッシュに保存（24時間）
            set_transient($cache_key, $result, 24 * HOUR_IN_SECONDS);
        }
        
        return $result;
    }
    
    /**
     * 商品検索
     */
    public function search_products($keyword, $marketplace_code = null, $category = 'All', $page = 1) {
        // キャッシュチェック
        $cache_key = $this->get_cache_key('search', md5($keyword . $category . $page), $marketplace_code);
        $cached_data = get_transient($cache_key);
        if ($cached_data !== false) {
            return $cached_data;
        }
        
        // API呼び出し
        $result = $this->client->search_products($keyword, $marketplace_code, $category, $page);
        
        if (!is_wp_error($result)) {
            // キャッシュに保存（1時間）
            set_transient($cache_key, $result, HOUR_IN_SECONDS);
        }
        
        return $result;
    }
    
    /**
     * キャッシュキーの生成
     */
    private function get_cache_key($type, $identifier, $marketplace_code = null) {
        if ($marketplace_code === null) {
            $marketplace_code = $this->marketplace_manager->get_current_marketplace();
        }
        return sprintf('aab_%s_%s_%s', $type, $marketplace_code, $identifier);
    }
    
    /**
     * API状態通知を表示
     */
    public function display_api_status_notice() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        $screen = get_current_screen();
        if ($screen && $screen->id === 'settings_page_aab-settings') {
            return;
        }
        
        $notices = $this->error_handler->get_admin_notices();
        foreach ($notices as $notice) {
            echo $notice;
        }
    }
    
    /**
     * APIテスト
     */
    public function test_api_connection($marketplace_code = null) {
        if ($marketplace_code === null) {
            $marketplace_code = $this->marketplace_manager->get_current_marketplace();
        }
        
        // テスト用ASIN（各マーケットプレイスで存在する可能性が高い商品）
        $test_asins = array(
            'jp' => 'B08C1LR9RC',  // Echo Dot
            'us' => 'B08N5WRWNW',  // Echo Dot (US)
            'uk' => 'B07V1W9ZP3',  // Echo Dot (UK)
            'de' => 'B07PHPXHQS',  // Echo Dot (DE)
            'default' => 'B08C1LR9RC'
        );
        
        $test_asin = isset($test_asins[$marketplace_code]) ? 
                     $test_asins[$marketplace_code] : 
                     $test_asins['default'];
        
        $result = $this->get_product_info($test_asin, $marketplace_code);
        
        if (is_wp_error($result)) {
            return array(
                'success' => false,
                'message' => $result->get_error_message(),
                'code' => $result->get_error_code(),
                'marketplace' => $marketplace_code
            );
        }
        
        return array(
            'success' => true,
            'message' => 'API接続成功！',
            'product' => $result,
            'marketplace' => $marketplace_code
        );
    }
    
    /**
     * AJAX: 商品情報取得
     */
    public function ajax_get_product_info() {
        check_ajax_referer('aab_ajax_nonce', 'nonce');
        
        $asin = sanitize_text_field($_POST['asin'] ?? '');
        $marketplace = sanitize_text_field($_POST['marketplace'] ?? null);
        
        if (empty($asin)) {
            wp_send_json_error('ASINが指定されていません');
            return;
        }
        
        $product = $this->get_product_info($asin, $marketplace);
        
        if (is_wp_error($product)) {
            $message = $this->error_handler->get_user_friendly_message($product);
            wp_send_json_error($message);
            return;
        }
        
        wp_send_json_success($product);
    }
    
    /**
     * AJAX: 商品検索
     */
    public function ajax_search_products() {
        check_ajax_referer('aab_ajax_nonce', 'nonce');
        
        $keyword = sanitize_text_field($_POST['keyword'] ?? '');
        $marketplace = sanitize_text_field($_POST['marketplace'] ?? null);
        $category = sanitize_text_field($_POST['category'] ?? 'All');
        $page = intval($_POST['page'] ?? 1);
        
        if (empty($keyword)) {
            wp_send_json_error('検索キーワードが指定されていません');
            return;
        }
        
        $products = $this->search_products($keyword, $marketplace, $category, $page);
        
        if (is_wp_error($products)) {
            $message = $this->error_handler->get_user_friendly_message($products);
            wp_send_json_error($message);
            return;
        }
        
        wp_send_json_success($products);
    }
    
    /**
     * AJAX: APIテスト
     */
    public function ajax_test_api() {
        check_ajax_referer('aab_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
            return;
        }
        
        $marketplace = sanitize_text_field($_POST['marketplace'] ?? null);
        $result = $this->test_api_connection($marketplace);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    /**
     * AJAX: マーケットプレイス状態取得
     */
    public function ajax_get_marketplace_status() {
        check_ajax_referer('aab_ajax_nonce', 'nonce');
        
        $status = $this->marketplace_manager->get_all_marketplace_status();
        wp_send_json_success($status);
    }
    
    /**
     * 管理画面用スクリプトをエンキュー
     */
    public function enqueue_admin_scripts($hook) {
        if ('post.php' !== $hook && 'post-new.php' !== $hook && 'settings_page_aab-settings' !== $hook) {
            return;
        }
        
        $current_marketplace = $this->marketplace_manager->get_current_marketplace();
        $marketplace_status = $this->marketplace_manager->get_all_marketplace_status();
        
        wp_localize_script('aab-editor', 'aab_api', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('aab_ajax_nonce'),
            'current_marketplace' => $current_marketplace,
            'marketplace_status' => $marketplace_status,
            'api_status' => $this->api_status
        ));
    }
}