<?php
/**
 * 管理画面クラス
 */
class AAB_Admin {
    
    /**
     * コンストラクタ
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    /**
     * 管理メニュー追加
     */
    public function add_menu() {
        add_options_page(
            'Amazon Affiliate Block設定',
            'Amazon Affiliate',
            'manage_options',
            'aab-settings',
            array($this, 'settings_page')
        );
    }
    
    /**
     * 管理画面用スクリプト読み込み
     */
    public function enqueue_scripts($hook) {
        if ('settings_page_aab-settings' !== $hook) {
            return;
        }
        
        // 管理画面用のCSSがあれば読み込む
        $admin_css = AAB_PLUGIN_DIR . 'assets/css/admin.css';
        if (file_exists($admin_css)) {
            wp_enqueue_style(
                'aab-admin',
                AAB_PLUGIN_URL . 'assets/css/admin.css',
                array(),
                AAB_PLUGIN_VERSION
            );
        }
        
        // 管理画面用のJSがあれば読み込む
        $admin_js = AAB_PLUGIN_DIR . 'assets/js/admin.js';
        if (file_exists($admin_js)) {
            wp_enqueue_script(
                'aab-admin',
                AAB_PLUGIN_URL . 'assets/js/admin.js',
                array('jquery'),
                AAB_PLUGIN_VERSION,
                true
            );
        }
    }
    
    /**
     * 設定画面
     */
    public function settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('このページにアクセスする権限がありません。', 'amazon-affiliate-block'));
        }
        
        // 設定保存処理
        $this->save_settings();
        
        // 現在の設定値取得
        $settings = $this->get_settings();
        
        // ビューファイルを読み込む
        $view_file = AAB_PLUGIN_DIR . 'views/admin-settings.php';
        if (file_exists($view_file)) {
            include $view_file;
        } else {
            AAB_Helper::log_error('管理画面ビューファイルが見つかりません', array('file' => $view_file));
            echo '<div class="wrap"><h1>エラー</h1><p>設定画面の表示に失敗しました。</p></div>';
        }
    }
    
    /**
     * 設定保存
     */
    private function save_settings() {
        if (!isset($_POST['submit']) || !isset($_POST['aab_nonce'])) {
            return;
        }
        
        if (!wp_verify_nonce($_POST['aab_nonce'], 'aab_settings_save')) {
            add_settings_error('aab_settings', 'invalid_nonce', 'セキュリティチェックに失敗しました。', 'error');
            return;
        }
        
        try {
            // アフィリエイトタグ
            $affiliate_tag = sanitize_text_field($_POST['affiliate_tag'] ?? '');
            if (!empty($affiliate_tag)) {
                $validated_tag = AAB_Helper::validate_affiliate_tag($affiliate_tag);
                if (!$validated_tag) {
                    throw new Exception('アフィリエイトタグの形式が正しくありません。英数字とハイフンのみ使用可能です。');
                }
                $affiliate_tag = $validated_tag;
            }
            update_option('aab_affiliate_tag', $affiliate_tag);
            
            // API設定
            update_option('aab_api_key', sanitize_text_field($_POST['api_key'] ?? ''));
            update_option('aab_secret_key', sanitize_text_field($_POST['secret_key'] ?? ''));
            
            add_settings_error('aab_settings', 'settings_saved', '設定を保存しました。', 'success');
            
        } catch (Exception $e) {
            add_settings_error('aab_settings', 'save_error', $e->getMessage(), 'error');
            AAB_Helper::log_error('設定保存エラー', array('error' => $e->getMessage()));
        }
    }
    
    /**
     * 設定取得
     */
    private function get_settings() {
        $settings = array(
            'affiliate_tag' => get_option('aab_affiliate_tag', ''),
            'api_key' => get_option('aab_api_key', ''),
            'secret_key' => get_option('aab_secret_key', '')
        );
        
        // API設定状態を追加
        $settings['api_configured'] = !empty($settings['api_key']) && !empty($settings['secret_key']) && !empty($settings['affiliate_tag']);
        
        // レート制限状態を取得
        $rate_limit = get_transient('aab_api_rate_limit');
        $settings['rate_limit_active'] = $rate_limit !== false && $rate_limit >= 1;
        
        return $settings;
    }
    
    /**
     * 設定のエクスポート
     */
    public function export_settings() {
        $settings = array(
            'affiliate_tag' => get_option('aab_affiliate_tag'),
            'api_key' => get_option('aab_api_key'),
            'secret_key' => get_option('aab_secret_key'),
            'version' => AAB_PLUGIN_VERSION
        );
        
        return json_encode($settings, JSON_PRETTY_PRINT);
    }
    
    /**
     * 設定のインポート
     */
    public function import_settings($json_data) {
        try {
            $settings = json_decode($json_data, true);
            
            if (!is_array($settings)) {
                throw new Exception('無効な設定データです。');
            }
            
            // バージョンチェック
            if (isset($settings['version']) && version_compare($settings['version'], AAB_PLUGIN_VERSION, '>')) {
                throw new Exception('新しいバージョンの設定ファイルはインポートできません。');
            }
            
            // 設定の更新
            if (isset($settings['affiliate_tag'])) {
                update_option('aab_affiliate_tag', sanitize_text_field($settings['affiliate_tag']));
            }
            if (isset($settings['api_key'])) {
                update_option('aab_api_key', sanitize_text_field($settings['api_key']));
            }
            if (isset($settings['secret_key'])) {
                update_option('aab_secret_key', sanitize_text_field($settings['secret_key']));
            }
            
            return true;
            
        } catch (Exception $e) {
            AAB_Helper::log_error('設定インポートエラー', array('error' => $e->getMessage()));
            return false;
        }
    }
}