<?php
/**
 * 商品データパーサークラス
 * 
 * APIレスポンスのパースと正規化を担当
 */
class AAB_Product_Parser {
    
    /**
     * 商品情報レスポンスをパース
     */
    public function parse_product_response($response, $marketplace = null) {
        if (!isset($response['ItemsResult']['Items'][0])) {
            AAB_Helper::log_error('Product not found in response', array(
                'response' => json_encode($response)
            ));
            return null;
        }
        
        $item = $response['ItemsResult']['Items'][0];
        
        // 基本情報を抽出
        $product = array(
            'asin' => $this->extract_asin($item),
            'title' => $this->extract_title($item),
            'image' => $this->extract_image($item),
            'price' => $this->extract_price($item, $marketplace),
            'url' => $this->extract_url($item),
            'features' => $this->extract_features($item),
            'brand' => $this->extract_brand($item),
            'model' => $this->extract_model($item)
        );
        
        // マーケットプレイス情報を追加
        if ($marketplace) {
            $product['marketplace'] = array(
                'code' => $this->get_marketplace_code($marketplace),
                'name' => $marketplace['name'] ?? '',
                'currency' => $marketplace['currency'] ?? '',
                'currency_symbol' => $marketplace['currency_symbol'] ?? ''
            );
        }
        
        // フィルタリング（nullを除去）
        $product = array_filter($product, function($value) {
            return $value !== null && $value !== '';
        });
        
        return $product;
    }
    
    /**
     * 検索結果レスポンスをパース
     */
    public function parse_search_response($response, $marketplace = null) {
        if (!isset($response['SearchResult']['Items'])) {
            AAB_Helper::log_error('No search results found', array(
                'response' => json_encode($response)
            ));
            return array();
        }
        
        $products = array();
        
        foreach ($response['SearchResult']['Items'] as $item) {
            $product = array(
                'asin' => $this->extract_asin($item),
                'title' => $this->extract_title($item),
                'image' => $this->extract_image($item, 'Small'),
                'price' => $this->extract_price($item, $marketplace),
                'url' => $this->extract_url($item)
            );
            
            // フィルタリング
            $product = array_filter($product, function($value) {
                return $value !== null && $value !== '';
            });
            
            if (!empty($product['asin'])) {
                $products[] = $product;
            }
        }
        
        return $products;
    }
    
    /**
     * ASINを抽出
     */
    private function extract_asin($item) {
        return isset($item['ASIN']) ? $item['ASIN'] : null;
    }
    
    /**
     * タイトルを抽出
     */
    private function extract_title($item) {
        // タイトルの複数の場所を確認
        $paths = array(
            array('ItemInfo', 'Title', 'DisplayValue'),
            array('ItemInfo', 'Title', 'Label'),
            array('ItemAttributes', 'Title')
        );
        
        foreach ($paths as $path) {
            $value = $this->get_nested_value($item, $path);
            if ($value !== null) {
                return $this->clean_text($value);
            }
        }
        
        return null;
    }
    
    /**
     * 画像URLを抽出
     */
    private function extract_image($item, $size = 'Large') {
        // 画像の複数の場所を確認
        $paths = array(
            array('Images', 'Primary', $size, 'URL'),
            array('Images', 'Primary', $size, 'Url'),
            array('ImageSets', 'ImageSet', 0, $size . 'Image', 'URL'),
            array('LargeImage', 'URL')
        );
        
        foreach ($paths as $path) {
            $value = $this->get_nested_value($item, $path);
            if ($value !== null && $this->is_valid_url($value)) {
                return $value;
            }
        }
        
        // サイズが見つからない場合、他のサイズを試す
        if ($size === 'Large') {
            return $this->extract_image($item, 'Medium') ?: $this->extract_image($item, 'Small');
        }
        
        return null;
    }
    
    /**
     * 価格を抽出
     */
    private function extract_price($item, $marketplace = null) {
        // 価格の複数の場所を確認
        $paths = array(
            array('Offers', 'Listings', 0, 'Price', 'DisplayAmount'),
            array('Offers', 'Listings', 0, 'Price', 'Amount'),
            array('Offers', 'Summaries', 0, 'LowestPrice', 'DisplayAmount'),
            array('ItemInfo', 'ListPrice', 'DisplayAmount'),
            array('OfferSummary', 'LowestNewPrice', 'FormattedPrice')
        );
        
        foreach ($paths as $path) {
            $value = $this->get_nested_value($item, $path);
            if ($value !== null) {
                return $this->format_price($value, $marketplace);
            }
        }
        
        return null;
    }
    
    /**
     * URLを抽出
     */
    private function extract_url($item) {
        // URLの複数の場所を確認
        $paths = array(
            array('DetailPageURL'),
            array('ItemInfo', 'DetailPageURL'),
            array('ItemLinks', 'ItemLink', 0, 'URL')
        );
        
        foreach ($paths as $path) {
            $value = $this->get_nested_value($item, $path);
            if ($value !== null && $this->is_valid_url($value)) {
                return $value;
            }
        }
        
        return null;
    }
    
    /**
     * 商品特徴を抽出
     */
    private function extract_features($item) {
        $features = array();
        
        // Features配列を確認
        if (isset($item['ItemInfo']['Features'])) {
            if (isset($item['ItemInfo']['Features']['DisplayValues'])) {
                $features = $item['ItemInfo']['Features']['DisplayValues'];
            } elseif (is_array($item['ItemInfo']['Features'])) {
                $features = $item['ItemInfo']['Features'];
            }
        }
        
        // 文字列配列に正規化
        $features = array_map(function($feature) {
            return is_array($feature) ? (isset($feature['DisplayValue']) ? $feature['DisplayValue'] : '') : $feature;
        }, $features);
        
        return array_filter($features);
    }
    
    /**
     * ブランドを抽出
     */
    private function extract_brand($item) {
        $paths = array(
            array('ItemInfo', 'ByLineInfo', 'Brand', 'DisplayValue'),
            array('ItemInfo', 'ManufactureInfo', 'ItemPartNumber', 'DisplayValue'),
            array('ItemAttributes', 'Brand')
        );
        
        foreach ($paths as $path) {
            $value = $this->get_nested_value($item, $path);
            if ($value !== null) {
                return $this->clean_text($value);
            }
        }
        
        return null;
    }
    
    /**
     * モデル番号を抽出
     */
    private function extract_model($item) {
        $paths = array(
            array('ItemInfo', 'ManufactureInfo', 'Model', 'DisplayValue'),
            array('ItemInfo', 'ProductInfo', 'Model', 'DisplayValue'),
            array('ItemAttributes', 'Model')
        );
        
        foreach ($paths as $path) {
            $value = $this->get_nested_value($item, $path);
            if ($value !== null) {
                return $this->clean_text($value);
            }
        }
        
        return null;
    }
    
    /**
     * ネストされた配列から値を取得
     */
    private function get_nested_value($array, $path) {
        $current = $array;
        
        foreach ($path as $key) {
            if (is_array($current) && isset($current[$key])) {
                $current = $current[$key];
            } else {
                return null;
            }
        }
        
        return $current;
    }
    
    /**
     * テキストをクリーニング
     */
    private function clean_text($text) {
        // HTMLタグを除去
        $text = strip_tags($text);
        
        // エンティティをデコード
        $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        
        // 余分な空白を除去
        $text = preg_replace('/\s+/', ' ', $text);
        
        return trim($text);
    }
    
    /**
     * 価格をフォーマット
     */
    private function format_price($price, $marketplace = null) {
        // すでにフォーマット済みの価格の場合
        if (strpos($price, '$') !== false || 
            strpos($price, '¥') !== false || 
            strpos($price, '£') !== false ||
            strpos($price, '€') !== false ||
            strpos($price, '₹') !== false) {
            return $price;
        }
        
        // 数値のみの場合、通貨記号を追加
        if (is_numeric($price) && $marketplace) {
            $symbol = $marketplace['currency_symbol'] ?? '';
            return $symbol . number_format($price, 0);
        }
        
        return $price;
    }
    
    /**
     * URLの妥当性をチェック
     */
    private function is_valid_url($url) {
        return filter_var($url, FILTER_VALIDATE_URL) !== false;
    }
    
    /**
     * マーケットプレイスコードを取得
     */
    private function get_marketplace_code($marketplace) {
        $domain_to_code = array(
            'co.jp' => 'jp',
            'com' => 'us',
            'co.uk' => 'uk',
            'de' => 'de',
            'fr' => 'fr',
            'it' => 'it',
            'es' => 'es',
            'ca' => 'ca',
            'com.au' => 'au',
            'in' => 'in'
        );
        
        if (isset($marketplace['domain'])) {
            return isset($domain_to_code[$marketplace['domain']]) ? 
                   $domain_to_code[$marketplace['domain']] : 'unknown';
        }
        
        return 'unknown';
    }
    
    /**
     * バッチ処理用：複数商品を一度にパース
     */
    public function parse_batch_response($response, $marketplace = null) {
        $products = array();
        
        if (isset($response['ItemsResult']['Items'])) {
            foreach ($response['ItemsResult']['Items'] as $item) {
                $product = $this->parse_single_item($item, $marketplace);
                if ($product) {
                    $products[] = $product;
                }
            }
        }
        
        return $products;
    }
    
    /**
     * 単一商品をパース（内部用）
     */
    private function parse_single_item($item, $marketplace = null) {
        $product = array(
            'asin' => $this->extract_asin($item),
            'title' => $this->extract_title($item),
            'image' => $this->extract_image($item),
            'price' => $this->extract_price($item, $marketplace),
            'url' => $this->extract_url($item)
        );
        
        // 必須フィールドチェック
        if (empty($product['asin'])) {
            return null;
        }
        
        return array_filter($product, function($value) {
            return $value !== null && $value !== '';
        });
    }
}