<?php
/**
 * マーケットプレイス管理クラス
 * 
 * 各国のAmazonマーケットプレイスの設定と管理を担当
 */
class AAB_Marketplace_Manager {
    
    private $marketplaces = array(
        'jp' => array(
            'name' => '日本',
            'domain' => 'co.jp',
            'host' => 'webservices.amazon.co.jp',
            'region' => 'us-west-2',
            'marketplace' => 'www.amazon.co.jp',
            'currency' => 'JPY',
            'currency_symbol' => '¥',
            'locale' => 'ja_JP'
        ),
        'us' => array(
            'name' => 'アメリカ',
            'domain' => 'com',
            'host' => 'webservices.amazon.com',
            'region' => 'us-east-1',
            'marketplace' => 'www.amazon.com',
            'currency' => 'USD',
            'currency_symbol' => '$',
            'locale' => 'en_US'
        ),
        'uk' => array(
            'name' => 'イギリス',
            'domain' => 'co.uk',
            'host' => 'webservices.amazon.co.uk',
            'region' => 'eu-west-1',
            'marketplace' => 'www.amazon.co.uk',
            'currency' => 'GBP',
            'currency_symbol' => '£',
            'locale' => 'en_GB'
        ),
        'de' => array(
            'name' => 'ドイツ',
            'domain' => 'de',
            'host' => 'webservices.amazon.de',
            'region' => 'eu-west-1',
            'marketplace' => 'www.amazon.de',
            'currency' => 'EUR',
            'currency_symbol' => '€',
            'locale' => 'de_DE'
        ),
        'fr' => array(
            'name' => 'フランス',
            'domain' => 'fr',
            'host' => 'webservices.amazon.fr',
            'region' => 'eu-west-1',
            'marketplace' => 'www.amazon.fr',
            'currency' => 'EUR',
            'currency_symbol' => '€',
            'locale' => 'fr_FR'
        ),
        'it' => array(
            'name' => 'イタリア',
            'domain' => 'it',
            'host' => 'webservices.amazon.it',
            'region' => 'eu-west-1',
            'marketplace' => 'www.amazon.it',
            'currency' => 'EUR',
            'currency_symbol' => '€',
            'locale' => 'it_IT'
        ),
        'es' => array(
            'name' => 'スペイン',
            'domain' => 'es',
            'host' => 'webservices.amazon.es',
            'region' => 'eu-west-1',
            'marketplace' => 'www.amazon.es',
            'currency' => 'EUR',
            'currency_symbol' => '€',
            'locale' => 'es_ES'
        ),
        'ca' => array(
            'name' => 'カナダ',
            'domain' => 'ca',
            'host' => 'webservices.amazon.ca',
            'region' => 'us-east-1',
            'marketplace' => 'www.amazon.ca',
            'currency' => 'CAD',
            'currency_symbol' => 'C$',
            'locale' => 'en_CA'
        ),
        'au' => array(
            'name' => 'オーストラリア',
            'domain' => 'com.au',
            'host' => 'webservices.amazon.com.au',
            'region' => 'us-west-2',
            'marketplace' => 'www.amazon.com.au',
            'currency' => 'AUD',
            'currency_symbol' => 'A$',
            'locale' => 'en_AU'
        ),
        'in' => array(
            'name' => 'インド',
            'domain' => 'in',
            'host' => 'webservices.amazon.in',
            'region' => 'eu-west-1',
            'marketplace' => 'www.amazon.in',
            'currency' => 'INR',
            'currency_symbol' => '₹',
            'locale' => 'en_IN'
        )
    );
    
    private $current_marketplace = 'jp';
    private $credentials = array();
    
    /**
     * コンストラクタ
     */
    public function __construct() {
        $this->current_marketplace = get_option('aab_default_marketplace', 'jp');
        $this->load_credentials();
    }
    
    /**
     * 認証情報を読み込み
     */
    private function load_credentials() {
        foreach ($this->marketplaces as $code => $marketplace) {
            // マーケットプレイス固有の設定を確認
            $access_key = get_option('aab_api_key_' . $code, '');
            $secret_key = get_option('aab_secret_key_' . $code, '');
            $partner_tag = get_option('aab_affiliate_tag_' . $code, '');
            
            // 固有設定がない場合はグローバル設定を使用
            if (empty($access_key)) {
                $access_key = get_option('aab_api_key', '');
            }
            if (empty($secret_key)) {
                $secret_key = get_option('aab_secret_key', '');
            }
            if (empty($partner_tag)) {
                // アフィリエイトクラスから取得
                if (class_exists('AAB_Affiliate')) {
                    $affiliate = new AAB_Affiliate();
                    $partner_tag = $affiliate->get_user_tag();
                } else {
                    $partner_tag = get_option('aab_affiliate_tag', '');
                }
            }
            
            $this->credentials[$code] = array(
                'access_key' => $access_key,
                'secret_key' => $secret_key,
                'partner_tag' => $partner_tag
            );
        }
    }
    
    /**
     * 現在のマーケットプレイスを取得
     */
    public function get_current_marketplace() {
        return $this->current_marketplace;
    }
    
    /**
     * マーケットプレイスを設定
     */
    public function set_current_marketplace($code) {
        if (isset($this->marketplaces[$code])) {
            $this->current_marketplace = $code;
            return true;
        }
        return false;
    }
    
    /**
     * マーケットプレイス情報を取得
     */
    public function get_marketplace($code = null) {
        if ($code === null) {
            $code = $this->current_marketplace;
        }
        
        if (!isset($this->marketplaces[$code])) {
            return null;
        }
        
        $marketplace = $this->marketplaces[$code];
        $marketplace['credentials'] = $this->credentials[$code];
        
        return $marketplace;
    }
    
    /**
     * すべてのマーケットプレイスを取得
     */
    public function get_all_marketplaces() {
        $all = array();
        foreach ($this->marketplaces as $code => $marketplace) {
            $all[$code] = array_merge($marketplace, array(
                'configured' => $this->is_configured($code)
            ));
        }
        return $all;
    }
    
    /**
     * マーケットプレイスが設定されているか確認
     */
    public function is_configured($code = null) {
        if ($code === null) {
            $code = $this->current_marketplace;
        }
        
        if (!isset($this->credentials[$code])) {
            return false;
        }
        
        $creds = $this->credentials[$code];
        
        // API制限状態をチェック
        $api_status = get_option('aab_api_status_' . $code, null);
        if ($api_status === 'throttled') {
            return false;
        }
        
        return !empty($creds['access_key']) && 
               !empty($creds['secret_key']) && 
               !empty($creds['partner_tag']);
    }
    
    /**
     * 認証情報を取得
     */
    public function get_credentials($code = null) {
        if ($code === null) {
            $code = $this->current_marketplace;
        }
        
        return isset($this->credentials[$code]) ? $this->credentials[$code] : null;
    }
    
    /**
     * すべてのマーケットプレイスの状態を取得
     */
    public function get_all_marketplace_status() {
        $status = array();
        
        foreach ($this->marketplaces as $code => $marketplace) {
            $status[$code] = array(
                'name' => $marketplace['name'],
                'configured' => $this->is_configured($code),
                'api_status' => get_option('aab_api_status_' . $code, 'unknown'),
                'has_credentials' => !empty($this->credentials[$code]['access_key'])
            );
        }
        
        return $status;
    }
    
    /**
     * Amazon URLを生成
     */
    public function generate_amazon_url($asin, $code = null, $custom_tag = null) {
        if ($code === null) {
            $code = $this->current_marketplace;
        }
        
        if (!isset($this->marketplaces[$code])) {
            return false;
        }
        
        $domain = $this->marketplaces[$code]['domain'];
        $tag = $custom_tag ?: $this->credentials[$code]['partner_tag'];
        
        if (empty($tag)) {
            AAB_Helper::log_error('アフィリエイトタグが設定されていません', array('marketplace' => $code));
            return false;
        }
        
        return sprintf('https://www.amazon.%s/dp/%s?tag=%s', $domain, $asin, $tag);
    }
    
    /**
     * マーケットプレイス設定を保存
     */
    public function save_marketplace_settings($code, $settings) {
        if (!isset($this->marketplaces[$code])) {
            return false;
        }
        
        // 認証情報を保存
        if (isset($settings['access_key'])) {
            update_option('aab_api_key_' . $code, sanitize_text_field($settings['access_key']));
        }
        if (isset($settings['secret_key'])) {
            update_option('aab_secret_key_' . $code, sanitize_text_field($settings['secret_key']));
        }
        if (isset($settings['partner_tag'])) {
            update_option('aab_affiliate_tag_' . $code, sanitize_text_field($settings['partner_tag']));
        }
        
        // 認証情報を再読み込み
        $this->load_credentials();
        
        return true;
    }
}