<?php
/**
 * エラーハンドリングクラス
 * 
 * APIエラーの処理とユーザー向けメッセージの生成を担当
 */
class AAB_Error_Handler {
    
    private $error_messages = array();
    private $admin_notices = array();
    
    /**
     * コンストラクタ
     */
    public function __construct() {
        $this->init_error_messages();
    }
    
    /**
     * エラーメッセージの初期化
     */
    private function init_error_messages() {
        $this->error_messages = array(
            'InvalidSignature' => array(
                'user' => '認証エラーが発生しました。管理者にお問い合わせください。',
                'admin' => 'APIキーまたはシークレットキーが正しくありません。設定を確認してください。',
                'action' => 'check_credentials'
            ),
            'InvalidPartnerTag' => array(
                'user' => 'アフィリエイトタグの設定に問題があります。',
                'admin' => 'アフィリエイトタグが無効です。Amazonアソシエイトで承認されているタグか確認してください。',
                'action' => 'check_partner_tag'
            ),
            'TooManyRequests' => array(
                'user' => '一時的にアクセスが制限されています。しばらく待ってから再試行してください。',
                'admin' => 'API制限に達しました。1分後に再試行してください。',
                'action' => 'wait_and_retry'
            ),
            'ItemNotFound' => array(
                'user' => '商品が見つかりませんでした。',
                'admin' => '指定されたASINの商品が存在しません。',
                'action' => 'check_asin'
            ),
            'InvalidParameterValue' => array(
                'user' => 'リクエストパラメータにエラーがあります。',
                'admin' => 'APIリクエストのパラメータが無効です。',
                'action' => 'check_parameters'
            )
        );
    }
    
    /**
     * APIエラーを処理
     */
    public function handle_api_error($error, $marketplace = null) {
        $error_code = $error['Code'] ?? 'unknown';
        $error_msg = $error['Message'] ?? 'Unknown error';
        
        // ログに記録
        $this->log_error($error_code, $error_msg, $marketplace);
        
        // 売上実績制限の特別処理
        if ($this->is_throttling_error($error_msg)) {
            return $this->handle_throttling_error($marketplace);
        }
        
        // 既知のエラーコード処理
        if (isset($this->error_messages[$error_code])) {
            $error_info = $this->error_messages[$error_code];
            
            // 管理者向け通知を追加
            if (current_user_can('manage_options')) {
                $this->add_admin_notice('error', $error_info['admin'], $marketplace);
            }
            
            return new WP_Error(
                $this->get_wp_error_code($error_code),
                $this->format_error_message($error_info['user'], $marketplace),
                array(
                    'original_code' => $error_code,
                    'original_message' => $error_msg,
                    'marketplace' => $marketplace,
                    'action' => $error_info['action']
                )
            );
        }
        
        // 未知のエラー
        return new WP_Error(
            'api_error',
            $this->format_error_message('APIエラーが発生しました: ' . $error_msg, $marketplace),
            array(
                'original_code' => $error_code,
                'original_message' => $error_msg,
                'marketplace' => $marketplace
            )
        );
    }
    
    /**
     * 売上実績制限エラーかチェック
     */
    private function is_throttling_error($error_msg) {
        $throttling_keywords = array(
            'request throttling',
            'denied due to request',
            'insufficient sales history',
            'sales performance'
        );
        
        foreach ($throttling_keywords as $keyword) {
            if (stripos($error_msg, $keyword) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * 売上実績制限エラーを処理
     */
    private function handle_throttling_error($marketplace = null) {
        $marketplace_name = $marketplace ? $marketplace['name'] : 'Amazon';
        
        // API状態を保存
        if ($marketplace) {
            $marketplace_code = array_search($marketplace['name'], array_column($this->get_marketplace_names(), 'name'));
            if ($marketplace_code !== false) {
                update_option('aab_api_status_' . $marketplace_code, 'throttled');
            }
        }
        
        // 管理者向け詳細通知
        if (current_user_can('manage_options')) {
            $this->add_throttling_notice($marketplace_name);
        }
        
        return new WP_Error(
            'throttled',
            sprintf(
                '【%sからの通知】売上実績が不足しているため、商品情報の自動取得ができません。商品情報は手動で入力してください。',
                $marketplace_name
            ),
            array(
                'marketplace' => $marketplace,
                'action' => 'manual_input_required',
                'is_bug' => false,
                'solution' => '3件以上の売上を発生させ、24-48時間待つと自動的に解除されます。'
            )
        );
    }
    
    /**
     * ユーザーフレンドリーなメッセージを取得
     */
    public function get_user_friendly_message($wp_error) {
        if (!is_wp_error($wp_error)) {
            return '';
        }
        
        $code = $wp_error->get_error_code();
        $data = $wp_error->get_error_data();
        
        // 売上実績制限の場合
        if ($code === 'throttled') {
            return '⚠️ Amazonの制限により自動取得できません。商品名を手動で入力してください。（これはバグではありません）';
        }
        
        // API未設定の場合
        if ($code === 'not_configured') {
            return 'PA-APIが設定されていません。管理画面から設定を行ってください。';
        }
        
        // その他のエラー
        return $wp_error->get_error_message();
    }
    
    /**
     * 管理画面通知を追加
     */
    private function add_admin_notice($type, $message, $marketplace = null) {
        $marketplace_name = $marketplace ? ' (' . $marketplace['name'] . ')' : '';
        
        $this->admin_notices[] = sprintf(
            '<div class="notice notice-%s is-dismissible"><p><strong>Amazon Affiliate Block%s:</strong> %s</p></div>',
            $type === 'error' ? 'error' : 'warning',
            $marketplace_name,
            esc_html($message)
        );
    }
    
    /**
     * 売上実績制限の通知を追加
     */
    private function add_throttling_notice($marketplace_name) {
        ob_start();
        ?>
        <div class="notice notice-warning is-dismissible">
            <h3>📢 Amazon Affiliate Block - PA-API利用制限のお知らせ</h3>
            <p><strong>これはバグではありません。</strong><?php echo esc_html($marketplace_name); ?>の仕様により、PA-API機能が制限されています。</p>
            
            <div style="background: #fff; padding: 10px; border-left: 4px solid #ff9800; margin: 10px 0;">
                <h4 style="margin-top: 0;">🔍 制限の理由</h4>
                <p>Amazonアソシエイトプログラムの規約により、<strong>過去30日間に3件以上の売上実績</strong>がないアカウントはPA-APIを利用できません。</p>
            </div>
            
            <div style="background: #f0f8ff; padding: 10px; border-left: 4px solid #2196F3; margin: 10px 0;">
                <h4 style="margin-top: 0;">✅ プラグインは正常に動作します</h4>
                <ol>
                    <li>ASINコードを入力</li>
                    <li>商品名を手動で入力</li>
                    <li>必要に応じて画像URLと価格も入力</li>
                </ol>
            </div>
            
            <p>
                <a href="<?php echo admin_url('options-general.php?page=aab-settings'); ?>" class="button button-primary">設定画面へ</a>
                <a href="https://affiliate.amazon.co.jp/" target="_blank" class="button">Amazonアソシエイト管理画面</a>
            </p>
        </div>
        <?php
        $this->admin_notices[] = ob_get_clean();
    }
    
    /**
     * 管理画面通知を取得
     */
    public function get_admin_notices() {
        return $this->admin_notices;
    }
    
    /**
     * エラーログに記録
     */
    private function log_error($code, $message, $marketplace = null) {
        $context = array(
            'code' => $code,
            'message' => $message
        );
        
        if ($marketplace) {
            $context['marketplace'] = $marketplace['name'];
        }
        
        AAB_Helper::log_error('API Error', $context);
    }
    
    /**
     * エラーメッセージをフォーマット
     */
    private function format_error_message($message, $marketplace = null) {
        if ($marketplace && isset($marketplace['name'])) {
            return sprintf('[%s] %s', $marketplace['name'], $message);
        }
        return $message;
    }
    
    /**
     * WordPressエラーコードを取得
     */
    private function get_wp_error_code($api_error_code) {
        $code_map = array(
            'InvalidSignature' => 'auth_error',
            'InvalidPartnerTag' => 'tag_error',
            'TooManyRequests' => 'rate_limit',
            'ItemNotFound' => 'not_found',
            'InvalidParameterValue' => 'param_error'
        );
        
        return isset($code_map[$api_error_code]) ? $code_map[$api_error_code] : 'api_error';
    }
    
    /**
     * マーケットプレイス名のリストを取得
     */
    private function get_marketplace_names() {
        return array(
            'jp' => array('name' => '日本'),
            'us' => array('name' => 'アメリカ'),
            'uk' => array('name' => 'イギリス'),
            'de' => array('name' => 'ドイツ'),
            'fr' => array('name' => 'フランス'),
            'it' => array('name' => 'イタリア'),
            'es' => array('name' => 'スペイン'),
            'ca' => array('name' => 'カナダ'),
            'au' => array('name' => 'オーストラリア'),
            'in' => array('name' => 'インド')
        );
    }
}