<?php
/**
 * AWS Signature V4 生成クラス
 * 
 * Amazon PA-API用の署名生成を担当
 */
class AAB_Api_Signature {
    
    /**
     * 署名を生成
     */
    public function generate($method, $path, $headers, $payload, $marketplace) {
        $timestamp = $headers['x-amz-date'];
        $date = substr($timestamp, 0, 8);
        
        // 署名生成
        return $this->create_aws_signature_v4(
            $method,
            $path,
            $payload,
            $timestamp,
            $headers,
            $marketplace
        );
    }
    
    /**
     * AWS Signature Version 4 生成
     */
    private function create_aws_signature_v4($method, $path, $payload, $timestamp, $headers, $marketplace) {
        $date = substr($timestamp, 0, 8);
        $service = 'ProductAdvertisingAPI';
        $credentials = $marketplace['credentials'];
        
        // 1. 正規リクエストの作成
        $canonical_request = $this->create_canonical_request($method, $path, $headers, $payload);
        
        // 2. 署名文字列の作成
        $credential_scope = $this->create_credential_scope($date, $marketplace['region'], $service);
        $string_to_sign = $this->create_string_to_sign($timestamp, $credential_scope, $canonical_request);
        
        // 3. 署名の計算
        $signature = $this->calculate_signature($string_to_sign, $credentials['secret_key'], $date, $marketplace['region'], $service);
        
        // 4. Authorization ヘッダーの作成
        $signed_headers = $this->get_signed_headers($headers);
        
        return sprintf(
            'AWS4-HMAC-SHA256 Credential=%s/%s, SignedHeaders=%s, Signature=%s',
            $credentials['access_key'],
            $credential_scope,
            $signed_headers,
            $signature
        );
    }
    
    /**
     * 正規リクエストを作成
     */
    private function create_canonical_request($method, $path, $headers, $payload) {
        // ヘッダーの正規化
        $canonical_headers = '';
        $signed_headers_arr = array();
        
        // ヘッダーをソート
        ksort($headers);
        
        foreach ($headers as $key => $value) {
            $lowercase_key = strtolower($key);
            $canonical_headers .= $lowercase_key . ':' . trim($value) . "\n";
            $signed_headers_arr[] = $lowercase_key;
        }
        
        $signed_headers = implode(';', $signed_headers_arr);
        
        // 正規リクエスト
        return implode("\n", array(
            $method,
            $path,
            '',  // クエリストリング（今回は空）
            $canonical_headers,
            $signed_headers,
            hash('sha256', $payload)
        ));
    }
    
    /**
     * クレデンシャルスコープを作成
     */
    private function create_credential_scope($date, $region, $service) {
        return sprintf('%s/%s/%s/aws4_request', $date, $region, $service);
    }
    
    /**
     * 署名文字列を作成
     */
    private function create_string_to_sign($timestamp, $credential_scope, $canonical_request) {
        return implode("\n", array(
            'AWS4-HMAC-SHA256',
            $timestamp,
            $credential_scope,
            hash('sha256', $canonical_request)
        ));
    }
    
    /**
     * 署名を計算
     */
    private function calculate_signature($string_to_sign, $secret_key, $date, $region, $service) {
        // 署名キーの生成
        $k_date = hash_hmac('sha256', $date, 'AWS4' . $secret_key, true);
        $k_region = hash_hmac('sha256', $region, $k_date, true);
        $k_service = hash_hmac('sha256', $service, $k_region, true);
        $k_signing = hash_hmac('sha256', 'aws4_request', $k_service, true);
        
        // 最終署名
        return hash_hmac('sha256', $string_to_sign, $k_signing);
    }
    
    /**
     * 署名対象ヘッダーを取得
     */
    private function get_signed_headers($headers) {
        $signed_headers = array();
        foreach ($headers as $key => $value) {
            $signed_headers[] = strtolower($key);
        }
        sort($signed_headers);
        return implode(';', $signed_headers);
    }
    
    /**
     * デバッグ用：署名プロセスを検証
     */
    public function debug_signature($method, $path, $headers, $payload, $marketplace) {
        $debug_info = array();
        
        $timestamp = $headers['x-amz-date'];
        $date = substr($timestamp, 0, 8);
        $service = 'ProductAdvertisingAPI';
        
        // 正規リクエスト
        $canonical_request = $this->create_canonical_request($method, $path, $headers, $payload);
        $debug_info['canonical_request'] = $canonical_request;
        
        // クレデンシャルスコープ
        $credential_scope = $this->create_credential_scope($date, $marketplace['region'], $service);
        $debug_info['credential_scope'] = $credential_scope;
        
        // 署名文字列
        $string_to_sign = $this->create_string_to_sign($timestamp, $credential_scope, $canonical_request);
        $debug_info['string_to_sign'] = $string_to_sign;
        
        // 署名
        $signature = $this->calculate_signature(
            $string_to_sign, 
            $marketplace['credentials']['secret_key'], 
            $date, 
            $marketplace['region'], 
            $service
        );
        $debug_info['signature'] = $signature;
        
        return $debug_info;
    }
}