<?php
/**
 * Amazon PA-API クライアントクラス
 * 
 * 実際のAPI通信を担当
 */
class AAB_Api_Client {
    
    private $marketplace_manager;
    private $error_handler;
    private $signature_generator;
    private $product_parser;
    
    /**
     * コンストラクタ
     */
    public function __construct($marketplace_manager, $error_handler) {
        $this->marketplace_manager = $marketplace_manager;
        $this->error_handler = $error_handler;
        $this->signature_generator = new AAB_Api_Signature();
        $this->product_parser = new AAB_Product_Parser();
    }
    
    /**
     * 商品情報を取得
     */
    public function get_product($asin, $marketplace_code = null) {
        if ($marketplace_code !== null) {
            $this->marketplace_manager->set_current_marketplace($marketplace_code);
        }
        
        $marketplace = $this->marketplace_manager->get_marketplace();
        if (!$marketplace) {
            return new WP_Error('invalid_marketplace', 'マーケットプレイスが無効です');
        }
        
        if (!$this->marketplace_manager->is_configured()) {
            return new WP_Error('not_configured', 
                sprintf('%sのPA-APIが設定されていません', $marketplace['name']));
        }
        
        // リクエストペイロード
        $request = array(
            'ItemIds' => array($asin),
            'Resources' => array(
                'Images.Primary.Large',
                'ItemInfo.Title',
                'ItemInfo.Features',
                'ItemInfo.ProductInfo',
                'Offers.Listings.Price',
                'Offers.Listings.SavingBasis'
            ),
            'PartnerTag' => $marketplace['credentials']['partner_tag'],
            'PartnerType' => 'Associates',
            'Marketplace' => $marketplace['marketplace']
        );
        
        // API呼び出し
        $response = $this->call_api('GetItems', $request, $marketplace);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        // レスポンスをパース
        return $this->product_parser->parse_product_response($response, $marketplace);
    }
    
    /**
     * 商品検索
     */
    public function search_products($keyword, $marketplace_code = null, $category = 'All', $page = 1) {
        if ($marketplace_code !== null) {
            $this->marketplace_manager->set_current_marketplace($marketplace_code);
        }
        
        $marketplace = $this->marketplace_manager->get_marketplace();
        if (!$marketplace) {
            return new WP_Error('invalid_marketplace', 'マーケットプレイスが無効です');
        }
        
        if (!$this->marketplace_manager->is_configured()) {
            return new WP_Error('not_configured', 
                sprintf('%sのPA-APIが設定されていません', $marketplace['name']));
        }
        
        // リクエストペイロード
        $request = array(
            'Keywords' => $keyword,
            'SearchIndex' => $category,
            'ItemPage' => $page,
            'Resources' => array(
                'Images.Primary.Small',
                'ItemInfo.Title',
                'Offers.Listings.Price'
            ),
            'PartnerTag' => $marketplace['credentials']['partner_tag'],
            'PartnerType' => 'Associates',
            'Marketplace' => $marketplace['marketplace']
        );
        
        // API呼び出し
        $response = $this->call_api('SearchItems', $request, $marketplace);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        // レスポンスをパース
        return $this->product_parser->parse_search_response($response, $marketplace);
    }
    
    /**
     * API呼び出し
     */
    private function call_api($operation, $payload, $marketplace) {
        $path = '/paapi5/' . strtolower($operation);
        $target = 'com.amazon.paapi5.v1.ProductAdvertisingAPIv1.' . $operation;
        
        $payload_json = json_encode($payload);
        $timestamp = gmdate('Ymd\THis\Z');
        
        // ヘッダー準備
        $headers = array(
            'content-encoding' => 'amz-1.0',
            'content-type' => 'application/json; charset=utf-8',
            'host' => $marketplace['host'],
            'x-amz-date' => $timestamp,
            'x-amz-target' => $target
        );
        
        // 署名生成
        $authorization = $this->signature_generator->generate(
            'POST',
            $path,
            $headers,
            $payload_json,
            $marketplace
        );
        
        $headers['Authorization'] = $authorization;
        
        // デバッグログ
        if (defined('WP_DEBUG') && WP_DEBUG) {
            AAB_Helper::log_error('API Request', array(
                'operation' => $operation,
                'marketplace' => $marketplace['name'],
                'url' => 'https://' . $marketplace['host'] . $path
            ));
        }
        
        // HTTPリクエスト
        $response = wp_remote_post('https://' . $marketplace['host'] . $path, array(
            'headers' => $headers,
            'body' => $payload_json,
            'timeout' => 10,
            'sslverify' => true
        ));
        
        if (is_wp_error($response)) {
            AAB_Helper::log_error('HTTP通信エラー', array(
                'marketplace' => $marketplace['name'],
                'error' => $response->get_error_message()
            ));
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        // レスポンスをデコード
        $data = json_decode($body, true);
        
        // エラーチェック
        if (isset($data['Errors'])) {
            return $this->error_handler->handle_api_error(
                $data['Errors'][0], 
                $marketplace
            );
        }
        
        if ($response_code !== 200) {
            AAB_Helper::log_error('HTTPエラー', array(
                'code' => $response_code,
                'body' => substr($body, 0, 500)
            ));
            return new WP_Error('http_error', 'HTTPエラー: ' . $response_code);
        }
        
        return $data;
    }
}