(function (blocks, element, editor, components, data) {
  const el = element.createElement;
  const { registerBlockType } = blocks;
  const { InspectorControls, MediaUpload } = editor;
  const { PanelBody, TextControl, SelectControl, Button, Spinner, Notice } = components;
  const { useState, useEffect } = element;

  registerBlockType("aab/amazon-product", {
    title: "Amazon商品",
    icon: "cart",
    category: "widgets",
    attributes: {
      asin: {
        type: "string",
        default: "",
      },
      title: {
        type: "string",
        default: "",
      },
      imageUrl: {
        type: "string",
        default: "",
      },
      price: {
        type: "string",
        default: "",
      },
      linkType: {
        type: "string",
        default: "text",
      },
      marketplace: {
        type: "string",
        default: "co.jp",
      },
    },

    edit: function (props) {
      const { attributes, setAttributes } = props;
      const { asin, title, imageUrl, price, linkType, marketplace } = attributes;
      const [isLoading, setIsLoading] = useState(false);
      const [searchKeyword, setSearchKeyword] = useState("");
      const [searchResults, setSearchResults] = useState([]);
      const [isSearching, setIsSearching] = useState(false);
      const [error, setError] = useState("");

      // PA-APIが設定されているかチェック
      const isApiConfigured = window.aab_api && window.aab_api.is_configured;

      // ASINから商品情報を自動取得
      const fetchProductInfo = () => {
        if (!asin || asin.length !== 10) {
          setError("正しいASIN（10文字）を入力してください");
          return;
        }

        if (!isApiConfigured) {
          setError("PA-APIが設定されていません。手動で商品情報を入力してください。");
          return;
        }

        setIsLoading(true);
        setError("");

        jQuery.ajax({
          url: window.aab_api.ajax_url,
          type: "POST",
          data: {
            action: "aab_get_product_info",
            asin: asin,
            nonce: window.aab_api.nonce,
          },
          success: function (response) {
            if (response.success && response.data) {
              setAttributes({
                title: response.data.title || "",
                imageUrl: response.data.image || "",
                price: response.data.price || "",
              });
              setError("");
            } else {
              setError(response.data || "商品情報の取得に失敗しました");
            }
            setIsLoading(false);
          },
          error: function () {
            setError("通信エラーが発生しました");
            setIsLoading(false);
          },
        });
      };

      // 商品検索
      const searchProducts = () => {
        if (!searchKeyword) {
          setError("検索キーワードを入力してください");
          return;
        }

        if (!isApiConfigured) {
          setError("PA-APIが設定されていません");
          return;
        }

        setIsSearching(true);
        setError("");

        jQuery.ajax({
          url: window.aab_api.ajax_url,
          type: "POST",
          data: {
            action: "aab_search_products",
            keyword: searchKeyword,
            category: "All",
            page: 1,
            nonce: window.aab_api.nonce,
          },
          success: function (response) {
            if (response.success && response.data) {
              setSearchResults(response.data);
              if (response.data.length === 0) {
                setError("商品が見つかりませんでした");
              }
            } else {
              setError(response.data || "検索に失敗しました");
            }
            setIsSearching(false);
          },
          error: function () {
            setError("通信エラーが発生しました");
            setIsSearching(false);
          },
        });
      };

      // 検索結果から商品を選択
      const selectProduct = (product) => {
        setAttributes({
          asin: product.asin,
          title: product.title,
          imageUrl: product.image,
          price: product.price,
        });
        setSearchResults([]);
        setSearchKeyword("");
      };

      return el(
        "div",
        {},
        el(
          InspectorControls,
          {},
          el(
            PanelBody,
            { title: "商品設定", initialOpen: true },

            // ASIN入力と自動取得
            el(TextControl, {
              label: "ASIN",
              value: asin,
              onChange: (value) => setAttributes({ asin: value }),
              help: "Amazon商品のASINコードを入力",
            }),

            isApiConfigured &&
              el(
                Button,
                {
                  isPrimary: true,
                  isBusy: isLoading,
                  disabled: isLoading || !asin,
                  onClick: fetchProductInfo,
                  style: { marginBottom: "10px", width: "100%" },
                },
                isLoading ? "取得中..." : "商品情報を自動取得"
              ),

            // エラー表示
            error &&
              el(
                Notice,
                {
                  status: "error",
                  isDismissible: false,
                },
                error
              ),

            // 商品検索セクション
            isApiConfigured &&
              el(
                "div",
                { style: { marginTop: "20px", marginBottom: "20px" } },
                el("h3", {}, "商品検索"),
                el(TextControl, {
                  label: "キーワード",
                  value: searchKeyword,
                  onChange: setSearchKeyword,
                  placeholder: "検索キーワードを入力",
                }),
                el(
                  Button,
                  {
                    isSecondary: true,
                    isBusy: isSearching,
                    disabled: isSearching || !searchKeyword,
                    onClick: searchProducts,
                    style: { width: "100%" },
                  },
                  isSearching ? "検索中..." : "商品を検索"
                ),

                // 検索結果
                searchResults.length > 0 &&
                  el(
                    "div",
                    {
                      style: {
                        marginTop: "10px",
                        maxHeight: "300px",
                        overflowY: "auto",
                        border: "1px solid #ddd",
                        padding: "10px",
                      },
                    },
                    searchResults.map((product) =>
                      el(
                        "div",
                        {
                          key: product.asin,
                          style: {
                            padding: "10px",
                            border: "1px solid #eee",
                            marginBottom: "10px",
                            cursor: "pointer",
                          },
                          onClick: () => selectProduct(product),
                        },
                        el(
                          "div",
                          { style: { display: "flex", alignItems: "center" } },
                          product.image &&
                            el("img", {
                              src: product.image,
                              style: { width: "50px", height: "50px", marginRight: "10px" },
                            }),
                          el("div", {}, el("strong", {}, product.title), product.price && el("div", {}, product.price), el("small", {}, "ASIN: " + product.asin))
                        )
                      )
                    )
                  )
              ),

            el("hr", {}),

            // 手動入力フィールド
            el("h3", {}, "商品情報（手動入力）"),
            el(TextControl, {
              label: "商品名",
              value: title,
              onChange: (value) => setAttributes({ title: value }),
            }),
            el(TextControl, {
              label: "画像URL",
              value: imageUrl,
              onChange: (value) => setAttributes({ imageUrl: value }),
              help: "商品画像のURLを入力",
            }),
            el(TextControl, {
              label: "価格",
              value: price,
              onChange: (value) => setAttributes({ price: value }),
              help: "表示する価格（オプション）",
            }),

            el("hr", {}),

            // 表示設定
            el(SelectControl, {
              label: "マーケットプレイス",
              value: marketplace,
              options: [
                { label: "日本 (co.jp)", value: "co.jp" },
                { label: "アメリカ (com)", value: "com" },
                { label: "イギリス (co.uk)", value: "co.uk" },
                { label: "ドイツ (de)", value: "de" },
                { label: "フランス (fr)", value: "fr" },
              ],
              onChange: (value) => setAttributes({ marketplace: value }),
            }),
            el(SelectControl, {
              label: "表示タイプ",
              value: linkType,
              options: [
                { label: "テキストリンク", value: "text" },
                { label: "カード表示", value: "card" },
                { label: "画像のみ", value: "image" },
              ],
              onChange: (value) => setAttributes({ linkType: value }),
            })
          )
        ),
        el(
          "div",
          { className: "aab-editor-preview" },
          // プレビュー表示
          linkType === "card"
            ? el("div", { className: "aab-preview-card" }, imageUrl && el("img", { src: imageUrl, style: { width: "100px" } }), el("div", {}, el("h4", {}, title || "Amazon商品"), price && el("p", {}, price), el("button", { className: "button" }, "Amazonで見る")))
            : linkType === "image" && imageUrl
            ? el("img", { src: imageUrl, style: { maxWidth: "200px" } })
            : el("a", { href: "#" }, title || "Amazon商品リンク")
        )
      );
    },

    save: function () {
      // サーバーサイドレンダリングを使用
      return null;
    },
  });
})(window.wp.blocks, window.wp.element, window.wp.blockEditor || window.wp.editor, window.wp.components, window.wp.data);
